//
// X509CertificateCollectionTest.cs: Unit tests for X509CertificationCollection
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// Copyright (C) 2004 Novell (http://www.novell.com)
//

using System;
using System.Security.Cryptography.X509Certificates;
using System.Text;

using NUnit.Framework;

namespace MonoTests.System.Security.Cryptography.X509Certificates {

	[TestFixture]
        public class X509CertificateCollectionTest {

		private static byte[] cert_a = { 0x30,0x82,0x01,0xFF,0x30,0x82,0x01,0x6C,0x02,0x05,0x02,0x72,0x00,0x06,0xE8,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,
			0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x37,0x5A,0x17,0x0D,0x39,0x37,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x36,0x5A,0x30,0x61,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x31,0x14,0x30,0x12,0x06,0x03,
			0x55,0x04,0x0A,0x13,0x0B,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x0B,0x13,0x1E,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x70,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x5F,0x00,0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,
			0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x54,0x20,0x67,0x12,0xBB,0x66,0x14,0xC3,0x26,0x6B,0x7F,
			0xDA,0x4A,0x25,0x4D,0x8B,0xE0,0xFD,0x1E,0x53,0x6D,0xAC,0xA2,0xD0,0x89,0xB8,0x2E,0x90,0xA0,0x27,0x43,0xA4,0xEE,0x4A,0x26,0x86,0x40,0xFF,0xB8,0x72,0x8D,0x1E,0xE7,0xB7,0x77,0xDC,0x7D,0xD8,0x3F,0x3A,0x6E,0x55,0x10,0xA6,0x1D,0xB5,0x58,0xF2,0xF9,0x0F,0x2E,0xB4,0x10,0x55,0x48,0xDC,0x13,0x5F,0x0D,0x08,0x26,0x88,0xC9,0xAF,0x66,0xF2,0x2C,0x9C,0x6F,0x3D,0xC3,0x2B,0x69,0x28,0x89,0x40,0x6F,0x8F,0x35,0x3B,0x9E,0xF6,0x8E,0xF1,0x11,0x17,0xFB,0x0C,0x98,0x95,0xA1,0xC2,0xBA,0x89,0x48,0xEB,0xB4,0x06,0x6A,0x22,0x54,
			0xD7,0xBA,0x18,0x3A,0x48,0xA6,0xCB,0xC2,0xFD,0x20,0x57,0xBC,0x63,0x1C };
		private static byte[] cert_b = { 0x30,0x82,0x01,0xDF,0x30,0x82,0x01,0x48,0x02,0x01,0x00,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x31,0x31,0x38,0x32,0x31,0x30,0x33,0x35,0x32,
			0x5A,0x17,0x0D,0x30,0x31,0x30,0x31,0x31,0x36,0x32,0x31,0x30,0x33,0x35,0x32,0x5A,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x81,0x9D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8B,0x00,0x30,0x81,0x87,
			0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,0xB8,0xA4,
			0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x38,0x50,0x1D,0x0A,0xD3,0x1B,0xBB,0xA2,0x9F,0x6C,0x8D,0x10,0xAA,0x42,0x1B,0x05,0x8D,0xE4,0x25,0xAB,0xFB,0x55,0xAE,0x6D,0xBA,0x53,0x67,0x15,0x07,0x9A,0xEC,0x55,0x9F,0x72,0x89,0x5F,0x24,0xB0,0xDB,0xCA,0x64,0xBD,0x64,0xAA,0xC2,0x8C,
			0xD9,0x3D,0xA2,0x45,0xB7,0xC6,0x92,0x71,0x51,0xEF,0xED,0xE1,0x51,0x54,0x97,0x56,0x35,0xA1,0xCE,0xE4,0x44,0xC4,0x47,0x66,0xFF,0x91,0xDA,0x88,0x9C,0x23,0xC2,0xB3,0xD4,0x62,0x4A,0xBC,0x94,0x55,0x9C,0x80,0x8E,0xB3,0xDD,0x4F,0x1A,0xED,0x12,0x5A,0xB5,0x2E,0xBC,0xF8,0x4B,0xCE,0xC6,0xD4,0x70,0xB3,0xB3,0x22,0xF8,0x5E,0x5C,0x36,0x7A,0xA6,0xB8,0x39,0x73,0x46,0x43,0x5C,0x9B,0x9A,0xBD,0x1E,0x7E,0xA7,0x04,0xCF,0x25,0x36 };
		private static byte[] cert_c = { 0x30,0x82,0x03,0x4E,0x30,0x82,0x02,0xB7,0xA0,0x03,0x02,0x01,0x02,0x02,0x20,0x03,0x53,0xD7,0x8B,0xDB,0x3E,0x16,0x15,0x80,0x55,0xC4,0x05,0x40,0x02,0x73,0x4D,0x0C,0x20,0xF8,0x0D,0x88,0x00,0x5F,0x65,0x7A,0xAC,0xBA,0x86,0xBD,0x1C,0xD7,0xE4,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x49,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x43,0x43,0x41,0x20,0x2D,0x20,0x55,0x6E,0x69,0x71,0x75,0x65,0x20,
			0x49,0x44,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x03,0x13,0x17,0x42,0x72,0x61,0x6E,0x64,0x20,0x4E,0x61,0x6D,0x65,0x3A,0x50,0x72,0x6F,0x64,0x75,0x63,0x74,0x20,0x54,0x79,0x70,0x65,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x38,0x30,0x37,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x39,0x36,0x30,0x38,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5A,0x30,0x6E,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x0A,0x13,0x07,0x42,0x72,0x61,0x6E,0x64,0x49,
			0x44,0x31,0x26,0x30,0x24,0x06,0x03,0x55,0x04,0x0B,0x13,0x1D,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x46,0x69,0x6E,0x61,0x6E,0x63,0x69,0x61,0x6C,0x20,0x49,0x6E,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x31,0x25,0x30,0x23,0x06,0x03,0x55,0x04,0x03,0x13,0x1C,0x30,0x2B,0x57,0x4B,0x4A,0x78,0x2B,0x77,0x59,0x45,0x5A,0x61,0x62,0x53,0x53,0x50,0x56,0x58,0x39,0x6B,0x4C,0x73,0x6E,0x78,0x39,0x32,0x73,0x3D,0x30,0x5C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x4B,0x00,
			0x30,0x48,0x02,0x41,0x00,0xAC,0xC4,0x0E,0x05,0x25,0xBC,0xEA,0xEF,0x0C,0x22,0x7F,0xC4,0x0C,0x4A,0x69,0x31,0x00,0xF9,0x3F,0xE9,0xE1,0x6C,0x54,0x97,0x77,0x4E,0x18,0xC6,0x4A,0x95,0xE0,0xD4,0x58,0x29,0x5C,0x17,0x5D,0x1D,0x1E,0x56,0xBC,0x49,0x3D,0xE0,0xF9,0x9F,0xBB,0x01,0xF9,0x86,0xB6,0xA6,0x95,0xDD,0xE1,0x04,0x32,0x01,0x52,0x4E,0x8F,0x86,0x30,0xF7,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,0x01,0x44,0x30,0x82,0x01,0x40,0x30,0x5C,0x06,0x03,0x55,0x1D,0x23,0x04,0x55,0x30,0x53,0x81,0x4B,0x84,0x49,0x31,0x0B,0x30,
			0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x43,0x43,0x41,0x20,0x2D,0x20,0x55,0x6E,0x69,0x71,0x75,0x65,0x20,0x49,0x44,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x03,0x13,0x17,0x42,0x72,0x61,0x6E,0x64,0x20,0x4E,0x61,0x6D,0x65,0x3A,0x50,0x72,0x6F,0x64,0x75,0x63,0x74,0x20,0x54,0x79,0x70,0x65,0x82,0x04,0x32,0x06,0xAC,0x10,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x2E,0x06,0x03,0x55,0x1D,0x10,0x01,
			0x01,0xFF,0x04,0x24,0x30,0x22,0x80,0x0F,0x31,0x39,0x39,0x36,0x30,0x38,0x30,0x37,0x30,0x37,0x34,0x39,0x30,0x30,0x5A,0x81,0x0F,0x31,0x39,0x39,0x36,0x30,0x39,0x30,0x37,0x30,0x37,0x34,0x39,0x30,0x30,0x5A,0x30,0x18,0x06,0x03,0x55,0x1D,0x20,0x04,0x11,0x30,0x0F,0x30,0x0D,0x06,0x0B,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x30,0x0C,0x06,0x03,0x55,0x1D,0x13,0x01,0x01,0xFF,0x04,0x02,0x30,0x00,0x30,0x14,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x02,0x03,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,
			0x07,0x80,0x30,0x62,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x02,0x07,0x01,0x01,0xFF,0x04,0x52,0x30,0x50,0x04,0x14,0x33,0x39,0x38,0x32,0x33,0x39,0x38,0x37,0x32,0x33,0x37,0x38,0x39,0x31,0x33,0x34,0x39,0x37,0x38,0x32,0x30,0x09,0x06,0x05,0x2B,0x0D,0x03,0x02,0x1A,0x05,0x00,0x16,0x0F,0x74,0x65,0x72,0x73,0x65,0x20,0x73,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x1D,0x00,0x16,0x1A,0x67,0x65,0x74,0x73,0x65,0x74,0x2D,0x63,0x65,0x6E,0x74,0x65,0x72,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,
			0x6F,0x6D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x1D,0x6C,0x90,0xF2,0x02,0x10,0xD5,0xA0,0x7B,0xEB,0x07,0x14,0x1D,0xAE,0x2C,0xD5,0xB1,0x4C,0x23,0x30,0xB2,0x31,0x7F,0x96,0x3C,0xD1,0x41,0x11,0xEA,0x08,0x0D,0x80,0x5F,0x46,0x66,0x9D,0x0A,0xF2,0x91,0xE8,0x7C,0xCE,0xC0,0xAD,0xE6,0x96,0x19,0x9C,0x26,0xC3,0x4E,0xEB,0x6F,0xF4,0x4A,0x69,0x4C,0xFE,0x4C,0x76,0xB3,0x51,0xCA,0x44,0x03,0xFC,0xD4,0xF4,0xF9,0x32,0x2A,0xAE,0x03,0x1B,0x5F,0xA6,0xBF,0x16,0x61,
			0xA0,0x07,0x86,0x85,0xA7,0xD6,0x0D,0xEF,0x88,0x9B,0x2A,0xBA,0xB8,0xD4,0x5C,0x94,0x7C,0x63,0xE1,0xE0,0x69,0xEB,0x3D,0x2F,0xC1,0x71,0x56,0x47,0x55,0x4B,0xB8,0xFD,0xCD,0xD6,0xC7,0x6F,0xA7,0x47,0xE9,0x43,0x91,0x2E,0xCF,0x93,0xA4,0x05,0x18,0xC5,0x98 };

		private static X509Certificate x509a;
		private static X509Certificate x509b;
		private static X509Certificate x509c;
		private static X509Certificate[] range;

		[TestFixtureSetUp]
		public void CreateCertificates () 
		{
			x509a = new X509Certificate (cert_a);
			x509b = new X509Certificate (cert_b);
			x509c = new X509Certificate (cert_c);
			range = new X509Certificate [2];
			range [0] = x509a;
			range [1] = x509b;
		}

		[Test]
		public void Constructor ()
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			Assert.AreEqual (0, c.Count, "Count");
		}

		[Test]
		public void Constructor_CertificateArray ()
		{
			X509CertificateCollection c = new X509CertificateCollection (range);
			Assert.AreEqual (2, c.Count, "Count");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Constructor_CertificateArray_Null ()
		{
			X509Certificate[] array = null;
			new X509CertificateCollection (array);
		}

		[Test]
		public void Constructor_CertificateCollection ()
		{
			X509CertificateCollection coll = new X509CertificateCollection (range);
			X509CertificateCollection c = new X509CertificateCollection (coll);
			Assert.AreEqual (2, c.Count, "Count");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Constructor_CertificateCollection_Null ()
		{
			X509CertificateCollection coll = null;
			new X509CertificateCollection (coll);
		}

		[Test]
		public void Add ()
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			Assert.AreEqual (0, c.Add (x509a), "Add a");
			Assert.AreEqual (1, c.Count, "1");

			Assert.AreEqual (1, c.Add (x509a), "Add a(dup)");
			Assert.AreEqual (2, c.Count, "2");

			Assert.AreEqual (2, c.Add (x509b), "Add b");
			Assert.AreEqual (3, c.Count, "3");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Add_Null ()
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			c.Add (null);
		}

		[Test]
		public void AddRange_Array ()
		{
			X509CertificateCollection c = new X509CertificateCollection ();

			c.AddRange (range);
			Assert.AreEqual (2, c.Count, "Range(a+b)");

			c.AddRange (range);
			Assert.AreEqual (4, c.Count, "Duplicate(a+b)");

			c.Add (x509c);
			Assert.AreEqual (5, c.Count, "New(c)");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void AddRange_ArrayNull ()
		{
			X509Certificate[] array = null;
			X509CertificateCollection c = new X509CertificateCollection ();
			c.AddRange (array);
		}

		[Test]
		public void AddRange_Collection ()
		{
			X509CertificateCollection coll = new X509CertificateCollection (range);
			X509CertificateCollection c = new X509CertificateCollection ();

			c.AddRange (c);
			Assert.AreEqual (0, c.Count, "Self(none)");

			c.AddRange (coll);
			Assert.AreEqual (2, c.Count, "Range(a+b)");

			c.AddRange (coll);
			Assert.AreEqual (4, c.Count, "Duplicate(a+b)");

			c.Add (x509c);
			Assert.AreEqual (5, c.Count, "New(c)");

			// This leads to an infinite loop until the runtime throws an OutOfMemoryException
			//c.AddRange (c);
			//Assert.AreEqual (10, c.Count, "Self(double)");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void AddRange_CollectionNull ()
		{
			X509CertificateCollection coll = null;
			X509CertificateCollection c = new X509CertificateCollection ();
			c.AddRange (coll);
		}

		[Test]
		public void Contains () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			Assert.IsTrue (!c.Contains (x509a), "Empty-A");
			Assert.IsTrue (!c.Contains (null), "Empty-Null");

			c.Add (x509a);
			Assert.IsTrue (c.Contains (x509a), "A-A");
			Assert.IsTrue (!c.Contains (x509b), "A-B");

			// works by value not by object reference
			X509Certificate x = new X509Certificate (cert_a);
			Assert.IsTrue (c.Contains (x), "A-x");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void CopyTo_Null () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			c.CopyTo (null, 0);
		}

		[Test]
		[ExpectedException (typeof (ArgumentException))]
		public void CopyTo_TooSmall () 
		{
			X509Certificate[] array = new X509Certificate [1];
			X509CertificateCollection c = new X509CertificateCollection ();
			c.AddRange (range);
			c.CopyTo (array, 0);
		}

		[Test]
		[ExpectedException (typeof (ArgumentOutOfRangeException))]
		public void CopyTo_IndexNegative () 
		{
			X509Certificate[] array = new X509Certificate [1];
			X509CertificateCollection c = new X509CertificateCollection ();
			c.CopyTo (array, -1);
		}

		[Test]
		public void CopyTo () 
		{
			X509Certificate[] array = new X509Certificate [1];
			X509CertificateCollection c = new X509CertificateCollection ();
			c.Add (x509a);
			c.CopyTo (array, 0);
			Assert.IsTrue (x509a.Equals (array [0]), "CopyTo");
		}

		[Test]
		public void IndexOf () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			Assert.AreEqual (-1, c.IndexOf (x509a), "Empty-A");
			Assert.AreEqual (-1, c.IndexOf (null), "Empty-Null");

			c.Add (x509a);
			Assert.AreEqual (0, c.IndexOf (x509a), "A-A");
			Assert.AreEqual (-1, c.IndexOf (x509b), "A-B");

			// works by object reference (not value)
			X509Certificate x = new X509Certificate (cert_a);
			Assert.IsTrue (!Object.ReferenceEquals (x509a, x), "!ReferenceEquals");
#if NET_2_0
			Assert.AreEqual (0, c.IndexOf (x), "A-x");
#else
			Assert.AreEqual (-1, c.IndexOf (x), "A-x");
#endif
		}

		[Test]
		public void Insert () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			c.Add (x509a);
			Assert.AreEqual (0, c.IndexOf (x509a), "a=0");
			c.Add (x509c);
			Assert.AreEqual (1, c.IndexOf (x509c), "c=1");

			c.Insert (1, x509b);
			Assert.AreEqual (1, c.IndexOf (x509b), "1");
			Assert.AreEqual (2, c.IndexOf (x509c), "2");
		}

		[Test]
		public void Remove () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			c.Add (x509a);
			Assert.AreEqual (1, c.Count, "Count==1");
			c.Remove (x509a);
			Assert.AreEqual (0, c.Count, "Remove Count==0");
		}

		[Test]
		[ExpectedException (typeof (ArgumentException))]
		public void Remove_Unexisting () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			// non existing
			c.Remove (x509a);
		}

		[Test]
#if !NET_2_0
		[ExpectedException (typeof (ArgumentException))]
#endif
		public void Remove_ByValue () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			X509Certificate x = null;

			try {
				// don't fail in this block
				c.Add (x509a);
				Assert.AreEqual (1, c.Count, "Read Count==1");

				// works by object reference (not by value)
				x = new X509Certificate (cert_a);
				Assert.IsTrue (!Object.ReferenceEquals (x509a, x), "!ReferenceEquals");
			}
			catch {}

			// fail here! (well for 1.x)
			c.Remove (x);
			Assert.AreEqual (0, c.Count, "Remove-by-value Count==0");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Remove_Null () 
		{
			X509CertificateCollection c = new X509CertificateCollection ();
			// non existing
			c.Remove (null);
		}
	}
}
