/*****************************************************************************
 *
 *  Copyright 2010 Richard Hacker (lerichi at gmx dot net)
 *
 *  This file is part of the pdserv library.
 *
 *  The pdserv library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published
 *  by the Free Software Foundation, either version 3 of the License, or (at
 *  your option) any later version.
 *
 *  The pdserv library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 *  License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with the pdserv library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#ifndef MAIN_H
#define MAIN_H

#include <map>
#include <set>
#include <list>
#include <memory>
#include <string>
#include <vector>
#include <stdint.h>
#include <log4cplus/logger.h>

#include "PThread.h"
#include "Config.h"
#include "Event.h"
#include "config.h"

#ifdef GNUTLS_FOUND
#include <gnutls/gnutls.h>
#include "TLS.h"
#endif

struct timespec;
class TlsSessionDB;

namespace MsrProto {
    class Server;
}

struct Blacklist;
class TlsSessionDB;

namespace PdServ {

class Signal;
class Event;
class Parameter;
class ProcessParameter;
class Variable;
class Task;
class Session;
struct SessionStatistics;

class Main {
    public:
        struct RtProcessExited {};

        Main(const std::string& name, const std::string& version);
        virtual ~Main();

        const std::string name;         /**< Application name */
        const std::string version;      /**< Application version */

        void startServers();
        void stopServers();

        bool loginMandatory() const;

#ifdef GNUTLS_FOUND
        bool tlsReady() const;
        void initTlsSessionData(gnutls_session_t) const;
        const Blacklist& getBlacklist() const { return blacklist; }
#endif

        // Get the current system time.
        // Reimplement this method in the class specialization
        virtual int gettime(struct timespec *) const;

        virtual std::list<const Task*> getTasks() const = 0;
        virtual std::list<const Event*> getEvents() const = 0;
        virtual std::list<const Parameter*> getParameters() const = 0;
        virtual void prepare(Session *session) const;
        virtual void cleanup(const Session *session) const;
        virtual Config config(const char* section) const = 0;

        void getActiveEvents(std::list<EventData>*) const;
        EventData getEvent(uint32_t id) const;
        uint32_t getCurrentEventId() const;

        // Setting a parameter has various steps:
        // 1) client calls parameter->setValue(session, ...)
        //    This virtual method is implemented by ProcessParameter
        // 2) ProcessParameter calls
        //        main->setValue(processParameter, session, ...)
        //    so that main can check whether session is allowed to set it
        // 3) main calls Main::setValue(...) virtual method to do the setting
        // 4) on success, main can do various functions, e.g.
        //      - inform clients of a value change
        //      - save persistent parameter
        //      - etc
        int setValue(const ProcessParameter* p, const Session *session,
                const char* buf, size_t offset, size_t count);

    protected:
        int setupLogging(const std::string& configFile);

        static int localtime(struct timespec *);

        void savePersistent();
        unsigned int setupPersistent();

        void newEvent(const Event* event,
                size_t element, int state, const struct timespec* time);

        // virtual functions to be implemented in derived classes
        virtual void initializeParameter(Parameter* p,
                const char* data, const struct timespec* mtime,
                const Signal* s) = 0;
        virtual bool getPersistentSignalValue(const Signal *s,
                char* buf, struct timespec* time) = 0;
        virtual Parameter* findParameter(const std::string& path) const = 0;
        virtual int setValue(const ProcessParameter* p,
                const char* buf, size_t offset, size_t count,
                const char** value, const struct timespec**) = 0;

    private:
        struct EventInstance;

        EventInstance* eventPtr;
        std::vector<EventInstance> eventInstance;
        std::set<EventInstance*> danglingInstances;

        typedef std::vector<EventInstance*> EventInstanceVector;
        typedef std::map<const Event*, EventInstanceVector*> EventInstanceMap;
        EventInstanceMap eventMap;

        mutable pthread::RWLock eventMutex;
        uint32_t eventSeqNo;

        const log4cplus::Logger parameterLog;
        const log4cplus::Logger eventLog;

        bool persistentLogTraceOn;
        log4cplus::Logger persistentLog;
        log4cplus::Logger persistentLogTrace;
        PdServ::Config persistentConfig;
        typedef std::map<const Parameter*, const Signal*> PersistentMap;
        PersistentMap persistentMap;

        MsrProto::Server *msrproto;

        void consoleLogging();
        void syslogLogging();
        const EventInstance* m_getEventInstance(uint32_t seqNo) const;

        void setupAccessControl(const Config&);
        bool m_loginMandatory;
        bool m_write;

#ifdef GNUTLS_FOUND
        pthread::Mutex tls_mutex;
        bool verifyClient;
        std::unique_ptr<gnutls_certificate_credentials_st, TlsDeleter> tls;
        std::unique_ptr<gnutls_priority_st, TlsDeleter> priority_cache;
        std::unique_ptr<gnutls_dh_params_int, TlsDeleter> dh_params;

        mutable TlsSessionDB tlsSessionDB;
        Blacklist blacklist;

        int setupTLS(Config, log4cplus::Logger&);
        void destroyTLS();
        void loadTrustFile(log4cplus::Logger&, const char* cafile);
        void loadCrlFile(log4cplus::Logger&, const char* cafile);
        void parseCertConfigDir(log4cplus::Logger&, const char* path,
                void (Main::*func)(log4cplus::Logger&, const char*));
        void parseCertConfigItem(log4cplus::Logger&, Config config,
                void (Main::*func)(log4cplus::Logger&, const char*));
#endif
};

}
#endif // MAIN_H
