/******************************************************************************
 *
 * ISO16845 Compliance tests
 * Copyright (C) 2021-present Ondrej Ille
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this SW component and associated documentation files (the "Component"),
 * to use, copy, modify, merge, publish, distribute the Component for
 * educational, research, evaluation, self-interest purposes. Using the
 * Component for commercial purposes is forbidden unless previously agreed with
 * Copyright holder.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Component.
 *
 * THE COMPONENT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHTHOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE COMPONENT OR THE USE OR OTHER DEALINGS
 * IN THE COMPONENT.
 *
 * @author Ondrej Ille, <ondrej.ille@gmail.com>
 * @date 31.10.2020
 *
 *****************************************************************************/

/******************************************************************************
 *
 * @test ISO16845 8.5.12
 *
 * @brief The purpose of this test is to verify that a passive state IUT acting
 *        as a transmitter waits for 6 consecutive identical bit to complete
 *        its passive error flag.
 *
 * @version Classical CAN, CAN FD tolerant, CAN FD enabled
 *
 * Test variables:
 *      Classical CAN, CAN FD tolerant, CAN FD enabled:
 *          FDF = 0
 *      CAN FD enabled:
 *          FDF = 1
 *
 * Elementary test cases:
 *      There is one elementary test to perform.
 *          #1 During the error flag, the LT sends 5 dominant bits,
 *             5 recessive bits and then, 6 dominant bits.
 *
 * Setup:
 *  The IUT is set to the TEC passive state
 *
 * Execution:
 *  The LT causes the IUT to transmit a frame.
 *  Then, the LT corrupts a bit in data field to cause the IUT to generate a
 *  passive error flag according to elementary test cases.
 *  After the 6 dominant bits, the LT waits for 8 bit time before sending a
 *  dominant bit.
 *
 * Response:
 *  The IUT shall generate an overload frame starting at the bit position
 *  following the last dominant bit generated by the LT.
 *
 *****************************************************************************/

#include <iostream>
#include <unistd.h>
#include <chrono>
#include <cmath>

#include "TestBase.h"

using namespace can;
using namespace test;

class TestIso_8_5_12 : public test::TestBase
{
    public:

        void ConfigureTest()
        {
            FillTestVariants(VariantMatchType::CommonAndFd);
            AddElemTest(TestVariant::Common, ElemTest(1, FrameKind::Can20));
            AddElemTest(TestVariant::CanFdEna, ElemTest(1, FrameKind::CanFd));

            /* Basic settings where IUT is transmitter */
            SetupMonitorTxTests();
            CanAgentConfigureTxToRxFeedback(true);

            /* To be error passive */
            dut_ifc->SetTec(160);
        }

        int RunElemTest([[maybe_unused]] const ElemTest &elem_test,
                        [[maybe_unused]] const TestVariant &test_variant)
        {
            uint8_t data_byte = 0x80;
            frm_flags = std::make_unique<FrameFlags>(elem_test.frame_kind_, IdentKind::Base,
                            RtrFlag::Data, BrsFlag::NoShift, EsiFlag::ErrPas);
            gold_frm = std::make_unique<Frame>(*frm_flags, 0x1, &data_byte);
            RandomizeAndPrint(gold_frm.get());

            drv_bit_frm = ConvBitFrame(*gold_frm);
            mon_bit_frm = ConvBitFrame(*gold_frm);

            /* Second frame the same due to retransmission. */
            drv_bit_frm_2 = ConvBitFrame(*gold_frm);
            mon_bit_frm_2 = ConvBitFrame(*gold_frm);

            /**************************************************************************************
             * Modify test frames:
             *   1. Force 7-th data bit to dominant to cause stuff error.
             *   2. Insert 5 dominant, 5 recessive and 6 dominant bits to driven frame right after
             *      bit with stuff error. Cut trailing bits. In monitored frame, insert 16
             *      recessive bits.
             *   3. Append 8 recessive bits to both driven and monitored frames.
             *   4. Insert single dominant bit to driven frame and recessive bit to monitored frame.
             *   5. Insert Overload frame to monitored frame after 8-th recessive bit. Insert
             *      Passive error frame to driven frame (TX/RX feedback enabled, just placeholder)
             *   6. Append retransmitted frame by IUT.
             *************************************************************************************/
            drv_bit_frm->GetBitOf(6, BitKind::Data)->FlipVal();

            drv_bit_frm->RemoveBitsFrom(7, BitKind::Data);
            mon_bit_frm->RemoveBitsFrom(7, BitKind::Data);

            for (size_t i = 0; i < 5; i++)
            {
                drv_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Dominant);
                mon_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Recessive);
            }

            // Compensate IUTs resynchronisation on first dominant bit caused by its input delay.
            mon_bit_frm->GetBitOf(0, BitKind::PasErrFlag)
                ->GetLastTQIter(BitPhase::Ph2)->Lengthen(dut_input_delay);

            for (size_t i = 0; i < 5; i++)
            {
                drv_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Recessive);
                mon_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Recessive);
            }

            for (size_t i = 0; i < 6; i++)
            {
                drv_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Dominant);
                mon_bit_frm->AppendBit(BitKind::PasErrFlag, BitVal::Recessive);
            }

            for (size_t i = 0; i < 8; i++)
            {
                drv_bit_frm->AppendBit(BitKind::ErrDelim, BitVal::Recessive);
                mon_bit_frm->AppendBit(BitKind::ErrDelim, BitVal::Recessive);
            }

            drv_bit_frm->AppendBit(BitKind::Interm, BitVal::Dominant);
            mon_bit_frm->AppendBit(BitKind::Interm, BitVal::Recessive);

            /*
             * Following bit is inserted just to be immediately overwritten by overload
             * frame!
             */
            drv_bit_frm->AppendBit(BitKind::Interm, BitVal::Recessive);
            mon_bit_frm->AppendBit(BitKind::Interm, BitVal::Recessive);

            drv_bit_frm->InsertOvrlFrm(1, BitKind::Interm);
            mon_bit_frm->InsertOvrlFrm(1, BitKind::Interm);

            drv_bit_frm->AppendSuspTrans();
            mon_bit_frm->AppendSuspTrans();

            drv_bit_frm_2->ConvRXFrame();
            drv_bit_frm->AppendBitFrame(drv_bit_frm_2.get());
            mon_bit_frm->AppendBitFrame(mon_bit_frm_2.get());

            drv_bit_frm->Print(true);
            mon_bit_frm->Print(true);

            /*****************************************************************************
             * Execute test
             *****************************************************************************/
            PushFramesToLT(*drv_bit_frm, *mon_bit_frm);
            StartDrvAndMon();
            this->dut_ifc->SendFrame(gold_frm.get());
            WaitForDrvAndMon();
            CheckLTResult();

            return FinishElemTest();
        }

};